package gov.va.vamf.scheduling.direct.datalayer.eligibility;

import gov.va.vamf.scheduling.direct.domain.CoreSetting;
import gov.va.vamf.scheduling.direct.domain.RequestEligibilityCriteria;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

@Service
public class RequestEligibilityCriteriaService {

    @Autowired
    private RequestEligibilityCriteriaRepository requestEligibilityCriteriaRepository;


    public RequestEligibilityCriteria retrieveRequestEligibilityCriteria(String siteCode) {
        return requestEligibilityCriteriaRepository.findById(siteCode);
    }

    public Iterable<RequestEligibilityCriteria> retrieveRequestEligibilityCriteriaByInstitutionCode(final Collection<String> institutionCodes) {
        return requestEligibilityCriteriaRepository.findAll(institutionCodes);
    }

    /**
     * Fetches one or more {@link gov.va.vamf.scheduling.direct.domain.RequestEligibilityCriteria}
     * @param ids the RequestEligibilityCriteria objects to fetch
     * @return a <code>Map</code> of ids to the requested criteria
     */
    public Map<String, RequestEligibilityCriteria> fetchAsMap(final Collection<String> ids) {
        final Map<String, RequestEligibilityCriteria> criteriaMap = new HashMap<>();

        final Iterable<RequestEligibilityCriteria> criteria = requestEligibilityCriteriaRepository.findAll(ids);

        for (final RequestEligibilityCriteria requestEligibilityCriteria : criteria) {
            criteriaMap.put(requestEligibilityCriteria.getId(), requestEligibilityCriteria);
        }

        return criteriaMap;
    }

    /**
     * Fetches a particular set of settings from a {@link gov.va.vamf.scheduling.direct.domain.RequestEligibilityCriteria}
     * @param criteriaId the RequestEligibilityCriteria to fetch
     * @param settingId the settings to fetch
     * @return
     */
    public Optional<CoreSetting> fetchSettings(final String criteriaId, final String settingId) {
        final RequestEligibilityCriteria criteria = retrieveRequestEligibilityCriteria(criteriaId);

        if (criteria == null) {
            return Optional.empty();
        }

        final Optional<CoreSetting> settings = criteria.getSetting(settingId);
        return settings;
    }

    /**
     * Fetches a particular set of settings from one or more {@link gov.va.vamf.scheduling.direct.domain.RequestEligibilityCriteria}
     * @param criteriaIds the RequestEligibilityCriteria objects to fetch
     * @param settingId the settings to fetch
     * @return a <code>Map</code> of ids to the requested critieria
     */
    public Map<String, CoreSetting> fetchSettingAsMap(final Collection<String> criteriaIds, final String settingId) {
        final Map<String, CoreSetting> settingsMap = new HashMap<>();

        Iterable<RequestEligibilityCriteria> criteria =
                retrieveRequestEligibilityCriteriaByInstitutionCode(criteriaIds);
        for (final RequestEligibilityCriteria criteriaEntry : criteria) {
            criteriaEntry.getSetting(settingId).ifPresent(new Consumer<CoreSetting>() { // TODO: switch to lambda
                @Override
                public void accept(final CoreSetting coreSetting) {
                    settingsMap.put(criteriaEntry.getId(), coreSetting);
                }
            });
        }

        return settingsMap;
    }
}
